/** @file   dynamicevent.h
 * @brief   Declaration of DynamicEvent - class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/07/27 20:27:34 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_DYNAMICEVENT_H
#define H_WWW_DYNAMICEVENT_H

#include <vector>
#include <iostream>



namespace WeWantWar {


/** @class  DynamicEvent
 * @brief   Baseclass of the event-classes such as SpawnEvent
 * @author  Tomi Lamminsaari
 *
 * Dynamic events provides the way to create events during the gameplay. These
 * events can be launched when player goes inside certain area. There is
 * also timer in each event that can delay the event.
 *
 * Each instance of DynamicEvent can have compound events that are launched
 * when the parent event is launched. Each of these inner events can have
 * their own delay when it actually happens after the parent event has occured.
 *
 * An instance of this DynamicEvent - class operates as container for the
 * derived events. As it triggers, it only activates the inner events,
 * such as @c SpawnEvent or @c ModifyEvent objects.
 *
 * The data of these events is loaded from level description file so you don't
 * need to create them from your code.
 */
class DynamicEvent
{
public:
  ///
  /// Static members and methods
  /// ==========================

  
  /** The possible statuses the events can have. */
  enum Status {
    /** This event is waiting for its turn */
    WAITING,
    /** This event is active and doing its operations. */
    ACTIVE,
    /** This event has been done and can be deleted */
    COMPLETE,
    /** This event has been deactivated. */
    SLEEP
  };
  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructor. */
  DynamicEvent();
  /** Copy constructor. */
  DynamicEvent(const DynamicEvent& rO);
  /** Destructor. */
  virtual ~DynamicEvent();
  /** Operator = */
  DynamicEvent& operator = (const DynamicEvent& rO);


  ///
  /// Methods
  /// =======


  /** Activates this event. Defined virtual since different events
   * need different implementation.
   */
  virtual void activate();
  
  /** Wakes up this event. Only events that are in SLEEP-mode can be
   * woken up.
   */
  virtual void wakeup();
  
  /** Sets this event sleeping
   */
  virtual void sleep();
  
  
  /** Reads the data of this event from given stream.
   * @param     fin               The input stream where we should read
   *                              the data.
   * @return    0 if successful. -1 if fails.
   */
  virtual int readData( std::istream& fin );
  
  /** Updates this event. If player goes inside the "active" region, this
   * method automatically activates itself and all the inner events.
   */
  virtual void update();
  
  /** Sets the status
   * @param     s                 New status
   */
  void status( Status s );
  
  /** Tells the status of this event.
   * @return    Current status. If COMPLETE, this event can be destroyed.
   */
  inline Status status() const { return m_status; }
  

protected:
  
  static const int KBitmaskEasy = 1;
  static const int KBitmaskNormal = 2;
  static const int KBitmaskHard = 4;
  
  /** Checks if this event can occure on current difficulty level.
   * @return  true if this event can occure on current difficulty level.
   */
  bool isOkToActivate() const;

  ///
  /// Members
  /// =======

  /** The status of this event. */
  Status  m_status;
  
  /** Counter that defines when this event actually happens after it has
   * been activated.
   */
  int     m_counter;
  
  /** Bitmask for difficulty levels where this event can occure. */
  int m_difficulty;
  
  /** The list of "inner" events that shoud happen as this event
   * happens.
   */
  std::vector<DynamicEvent*>  m_innerEvents;
  
  /** TopLeft X (in pixels) */
  int     m_x1;
  /** TopLeft Y (in pixels) */
  int     m_y1;
  /** BottomRight X (in pixels) */
  int     m_x2;
  /** BottomRight Y (in pixels) */
  int     m_y2;
  /** Index of the sound we should play when this event comes active.
   * -1 indicates that we should not make sound
   */
  int     m_sound;
};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: dynamicevent.h,v $
 * Revision 1.2  2006/07/27 20:27:34  lamminsa
 * New difficultylevel control.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:54+02  lamminsa
 * Initial revision
 *
 */
 

